function min_max_data = determine_spurts_significance( extrenum_data, pvalue, fdobj, stderr_data, degree )
    %
    % This function returns min_max_data with a new field 
    % to indicate if maxes in the degree'th derivative of the function 
    % are significant or not (other entries are tagged with 0)
    % The criterion used to determine significance based on 
    % confidence bands at the maxima (whether jerk is minimal) and the
    % neighbour points where jerk is maximal.
    % If the maxima band does not overlap with any of the neighbour points,
    % then the maxima is considered significant.
    %
    % Recall that confidence bands = mean +/- 1.96 * stderr_data for p=0.05
    % getBandWidthFactor(pvalue) returns the appropriate width for the p value
    % given
    %
    % Input parameters
    % ----------------
    % extrenum_data  a structure of information about the extrenum found. 
    %                Fields are: 
    %                * Location: indicates the x value where the extrenum was
    %                found
    %                * type indicate if it's a max or a min in the
    %                first (vel.) or second (acc.) curve (see global_constants
    %                for details)
    % fdobj         Functional object that fits the raw data
    % stderr_data   Vector of standard errors at the locations of the mins and
    %               max (in min_max_data)
    % degree        In which derivative are we looking for maxima and minima. To
    %               find max in the function, use degree = 0, and to find spurts (that is,
    %               max in velocity), use degree = 1
    %
    % Output values
    % -------------
    % min_max_data  The extrenum_data structure, expanded with an additional
    %               field:
    %               * significant that indicates if the extrenum is
    %               statistically significant or not. 
    %
    % TODO: Future improvement: Eventually, we might be interested in identifying 
    % significant minima as well.

    % loading globals
    global_constants;

    extrenum_data_size = size(extrenum_data, 2);

    % validation
    if (size(stderr_data,1) ~= extrenum_data_size)
        error('extrenum_data and stderr_data must have the same number of rows (entries)')
    end

    % extract x and y values for each min and max
    x_values = [extrenum_data.location];
    y_values = eval_fd(x_values, fdobj, degree);

    for extrenum_data_index=1:extrenum_data_size
        % determine what kind of min or max this is
        if (extrenum_data(extrenum_data_index).type == MAX_FCT)
            % OK, it's a maximum of the degree'th derivative (e.g. spurt)
            % (ignore other cases)

            % compute the lower bound of the confidence interval at the maximum
            point_low_conf_value = y_values(extrenum_data_index) - ...
                getBandWidthFactor(pvalue) * stderr_data(extrenum_data_index);

            % The two neighbour points (to compare with) are located two extrenums 
            % aside. Those points can be either:
            % (1) minima in the degree'th derivative
            % (2) inflexion point in the degree'th derivative curve - minima in 
            %     degree+1'th derivative (but degree'th derivative is still decreasing).
            % (points located one item aside correspond to inflexion points in
            % the degree+1'th derivative curve)

            % We use 2 events away, not 1, because basically, the best estimate of the two minima surrounding a 
            % maxima we have are located two "events" away (events are zero crossings in the velocity or 
            % acceleration curves):
            % Point - 2 : velo = "min", accel = 0
            % Point - 1 : velo = 0, accel = max;
            % Point (maxima): velo = max, accel = 0;
            % Point + 1 : velo = 0, accel = max;
            % Point + 2: velo = min, accel = 0;

            % A special case has to be made for when we are close to the
            % boundary of the range where the function is defined.

            % compute the upper bound of the confidence interval at the neighbour
            % point preceeding the maxima (spurt). We make a special case for the boundary
            % because we don't want the point to fall outside of the interval where the function is defined.
            % (if extrenum_data_index = 2, the function is defined at y_values(1), but not y_values(0)).
            if (extrenum_data_index > 2)
                prev_high_conf_value = y_values(extrenum_data_index - 2) + ...
                    getBandWidthFactor(pvalue) * stderr_data(extrenum_data_index-2);
            else
                % special case for boundary
                prev_high_conf_value = y_values(extrenum_data_index - 1) + ...
                    getBandWidthFactor(pvalue) * stderr_data(extrenum_data_index-1);
            end

            % compute the upper bound of the confidence interval at the
            % neighbour point following the maxima (spurt)
            if (extrenum_data_index < extrenum_data_size - 2)
                next_high_conf_value = y_values(extrenum_data_index + 2) + ...
                    getBandWidthFactor(pvalue) * stderr_data(extrenum_data_index+2);
            else
                % special case for boundary
                next_high_conf_value = y_values(extrenum_data_index + 1) + ...
                    getBandWidthFactor(pvalue) * stderr_data(extrenum_data_index+1);
            end

            % is the point max (spurt) statistically significant?
            % A maxima is significant if its lower confidence limit is higher
            % than its neighbours' upper ones.
            if ((point_low_conf_value > prev_high_conf_value) && (point_low_conf_value > next_high_conf_value))
                % reliable
                extrenum_data(extrenum_data_index).significant = SIGNIFICANT;
            else
                 % not reliable
                extrenum_data(extrenum_data_index).significant = NOT_SIGNIFICANT;
            end
        else
            extrenum_data(extrenum_data_index).significant = NOT_A_MAXIMA;
        end
    end

    min_max_data = extrenum_data;
return;