function clean_extrenum_data = eliminate_inflection_points(extrenum_data)
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % This function eliminates inflection points from the extrenum_data vector.
    % In the ideal case, the curve alternates between velocity
    % minima and maxima with minima and maxima of acceleration in between.
    %
    % However, in real curve, there might be some oscillations of the
    % acceleration (i.e., maxima and minima) that don't change the sign of the
    % velocity. That means the slope of the velocity curve changes, but it is
    % still increasing or decreasing.
    %
    % The idea is that when there 
    %
    % Input parameters
    % ----------------
    % extrenum_data  a structure of information about the extrenum found. 
    %                Fields are: 
    %                * location, where the extrenum event is located
    %                * type: type of extrenum (see global_constants)
    %
    % Output value
    % ------------
    % clean_extrenum_data  same as extrenum_data, but without inflection points
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    % loading global parameters
    global_constants;

    if (verbose)
        disp('original data')
        for i=1:size(extrenum_data,2)
            disp([num2str(i), ' location = ', num2str(extrenum_data(i).location), ' type = ', num2str(extrenum_data(i).type)])
        end
    end

    clean_extrenum_data = extrenum_data;

    % starting at the end of the vector (for the loop index to be valid)
    % look for trio of acceleration events. When found, drop the middle one.
    for i = (size(clean_extrenum_data,2)-1) : -1 : 2
         type1 = clean_extrenum_data(i+1).type;
         type2 = clean_extrenum_data(i).  type;
         type3 = clean_extrenum_data(i-1).type;

        % Normally, two velocity events should not occur consecutively
        if (((type1 == MAX_FCT) || (type1 == MIN_FCT)) && ...
            ((type2 == MAX_FCT) || (type2 == MIN_FCT)))
            %
            % If this condition occurs, it suggests that the data is too noisy
            % (and varies at too high frequency) to reliably detect velocity
            % and acceleration events (maxima and minima).
            %
            % After investigation of the data, you might decide to perform the
            % analysis anyway using die_on_error = false  
            %
            if (die_on_error)
                % exit with error message
                error('Error: Two consecutive velocity events')
            else
                % don't stop, just display an error message
                disp('Error: Two consecutive velocity events')
            end
        end

        % boundary case 1
        if ((abs(type1) == BOUNDARY) && ...
            (abs(type2) == MAX_DERIV) && ...
            (abs(type3) == MAX_DERIV))
                clean_extrenum_data(i) = [];
        end

        % boundary case 2
        if ((abs(type1) == MAX_DERIV) && ...
            (abs(type2) == MAX_DERIV) && ...
            (abs(type3) == BOUNDARY))
                clean_extrenum_data(i) = [];
        end

        % other cases
        if ((abs(type1) == MAX_DERIV) && ...
            (abs(type2) == MAX_DERIV) && ...
            (abs(type3) == MAX_DERIV))
                clean_extrenum_data(i) = [];
        end
    end

    if (verbose)
        disp('cleaned data')
        for i=1:size(clean_extrenum_data,2)
            disp([num2str(i), ' location = ', num2str(clean_extrenum_data(i).location), ' type = ', num2str(clean_extrenum_data(i).type)])
        end
    end
return;