function spurt_data = get_spurt_data_means_only( max_min_data, fdobj, degree, output_all_maxima )
    %
    % This function returns mean (expected value) of the estimated parameters for 
    % spurts in max_min_data. 
    %
    % Input parameters
    % ----------------
    % max_min_data - structure containing info on locations of x values where to find maxima and mimima 
    %                3 fields:
    %                * location: where extrenum is located (x value)
    %                * type: what kind of extrenum (see global_constants for details)
    %                * significant: if the extrenum is statistically
    %                significant
    % fdobj - the FDA fit of the raw data
    % degree - in which derivative we are looking for max and mins
    % output_all_maxima - set to true to generate all spurt data (including
    %                     non-significant one, or set to false to output significant data only
    %
    %
    % Output value
    % ------------
    % spurt_data: a structure with the following fields:
    % * begin_location - location of the beginning of the spurt
    % * center_location - Spurt center location (x value)
    % * amplitude - Spurt amplitude (or intensity)
    % * stderr_amplitude - Standard error on the amplitude estimate 
    % (One entry per maximum (spurt))

    % loading global constants
    global_constants;

    spurt_data = spurt_data_template;

    current_index = 1;            

    max_min_mat_size = size(max_min_data,2);

    x_values   = [max_min_data.location];
    vel_values = eval_fd(x_values, fdobj, degree+1);

    % for each element of the extrenum data matrix
    for min_max_data_index = 2:max_min_mat_size(1)-1
        % is this a function maximum?
        if (max_min_data(min_max_data_index).type == MAX_FCT)
            % process only if it's significant, or if we want all values (not
            % only significant ones)
            if ((max_min_data(min_max_data_index).significant == SIGNIFICANT) | output_all_maxima)

                % Computing means
                % Determine location and std error on location 
                % location = X
                % the spurt begins where the previous extrenum is located
                % (i.e., typically where acceleration is maximal)
                spurt_data(current_index).begin_location  = max_min_data(min_max_data_index-1).location;
                spurt_data(current_index).center_location = max_min_data(min_max_data_index).location;

                % Compute amplitude: difference between velocities at the
                % current location - velocity at the next mimimum
                % retrieve the location of the preceeding and following points
                % they correspond to the max, and min velocity values
                max_vel   = vel_values(min_max_data_index-1);
                min_vel   = vel_values(min_max_data_index+1);
                amplitude = max_vel - min_vel;

                % due to rounding errors, some very small maxima might get a 
                % negative value. They won't be significant anyway.
                if (amplitude < 0)
                    amplitude = 0.000001;
                end

                spurt_data(current_index).amplitude = amplitude;

                % Compute duration = x2 - x1
                duration = x_values(min_max_data_index+1) - x_values(min_max_data_index-1);
                spurt_data(current_index).duration = duration;

                current_index = current_index + 1;
            end
        end
    end
return;