function h = plot_spurts(spurts_data, simulation_parameters, sort_type)
%
% This function plots an array of spurts
% For each spurt in spurts_data, shows a line where the spurt is location
% (that is location +/- duration / 2) and the line thickness corresponds to
% the amplitude (intensity) of the spurt.
%
% Input parameters
% ----------------
% ncases: number of subjects or curves in the data
% spurts_data: see global_constants.m for details
% title_string: title to be displayed on the graph
% label_x: label of the x axis
% sort_type: sorting to do for presenting data (see global_constants for
%            details)
%
% Output values: None
% 
h = [];

% loading global constants;
global_constants;

if (size(spurts_data) == 0)
    disp('No spurts data to display (i.e., spurts_data is empty)')
    return
end

% find maximum spurt amplitude in the data
amp_max       = max([spurts_data.amplitude]);

% find extrenum x values to be displayed
x_max         = max([spurts_data.begin_location] + [spurts_data.duration]);
x_min         = min([spurts_data.duration]);

casesToDisplay = [];
for i=1:size(spurts_data,2)
    % add field to indicate if the data was plotted already or not
    spurts_data(i).bPlotted = 0;
    casesToDisplay = [casesToDisplay spurts_data(i).caseId];
%     size(spurts_data.caseId)
%     unique(spurts_data.caseId)
end
casesToDisplay = unique(casesToDisplay);
numberOfCasesToDisplay = size(casesToDisplay,2);

spacing = 20;

% line thickness for the largest amplitude to display
max_thickness = 0.45*spacing;

%plot the spurts_data
height = 180+spacing*numberOfCasesToDisplay;
h = figure('OuterPosition', [0, 0, 700, height], 'PaperPositionMode','auto'); %, 'visible', 'off');
hold on
title(simulation_parameters.title);
xlabel(simulation_parameters.x_label);
ylabel('Case id');
axis([x_min x_max 0 numberOfCasesToDisplay+1])

% converting structure to array for sorting purposes
spurts_data_matrix = [spurts_data.caseId; spurts_data.bPlotted; ...
                      spurts_data.begin_location; spurts_data.center_location; ...
                      spurts_data.amplitude; spurts_data.duration]';

if (sort_type == SORT_CASE_ID)
    spurts_data_matrix = sortrows(spurts_data_matrix, 1);
elseif (sort_type == SORT_BEGIN_LOCATION)
    spurts_data_matrix = sortrows(spurts_data_matrix, 3);
elseif (sort_type == SORT_AMPLITUDE)
    spurts_data_matrix = sortrows(spurts_data_matrix, 5);
else
    disp('invalid sort type');
    return
end

set(gcf, 'Color',[1,1,1])
yLabels = {''};
current_location = 1;

% display spurts for each curve
for j=1:size(spurts_data_matrix)
    % check if this subject_id was already processed
    if (spurts_data_matrix(j,2) == 0)
        % process next data point in whatever is left
        subject_id = spurts_data_matrix(j,1);

       % extracting spurts_data for this subject
       indices     = find (spurts_data_matrix(:,1) == subject_id);

       % display each spurt
       for i=1 : size(indices)
           index     = indices(i);
           center    = spurts_data_matrix (index,4);
           low_x     = spurts_data_matrix (index,3);
           high_x    = low_x + spurts_data_matrix (index,6);
           intensity = spurts_data_matrix(index,5) / amp_max;

           plot([low_x, high_x]',[current_location, current_location]', 'color', [0.5,0.5,0.5], 'LineWidth', intensity * max_thickness);
           text(center, current_location, num2str(i))  % display spurt number

           spurts_data_matrix(index,2) = 1;  % marking value as processed
       end
       % display case number
       yLabels(current_location+1) = {num2str(subject_id)};
       current_location = current_location + 1;
    end
end
yLabels(current_location+1) = {''};
set(gca,'ytick', 0:numberOfCasesToDisplay+1)
set(gca, 'YTickLabel', yLabels);
