% -----------------------------------------------------------------------
% Finds spurts and plateaus using AMD
%
% The spurt_analysis modules finds significant spurts in the data and, for
% each one, outputs 3 parameters (with std error values): amplitude,
% duration and location.
%
% This code can be customized to solve other problems.
% -----------------------------------------------------------------------

clear all
close all

addpath('../AMD_library')
addpath('../FDA_library')

% Loading global constants
global_constants;

lambda = 0.01;

resultsDir = './results/corrigan/';

mkdir(resultsDir);

% To set simulation parameters, see description of 
% simulation_parameters_template in
% file global_constants.m
simulation_parameters = simulation_parameters_template;

data = load('../data/corrigan/corrigan.dat');

sample_times = data(1,3:end)';
subject_ids = data(2:end,1);
group_ids = data(2:end,2);

data(:,1) = [];  % first column is subject id, so drop
data(:,1) = [];  % second column is group id, so drop
data(1,:) = [];  % first row is sample times, so drop

simulation_parameters(1).title = 'Vocabulary growth';
simulation_parameters.subject_ids = subject_ids;

simulation_parameters.sample_ts = sample_times;
simulation_parameters.x_label = 'Session id';

simulation_parameters.sample_ys = data';
simulation_parameters.y_label = 'Vocabulary item count';

simulation_parameters.group_ids = group_ids;
simulation_parameters.group_labels = {'Corrigan child'};

simulation_parameters.degree = 1;
simulation_parameters.pvalue = 0.05;
simulation_parameters.error_level   = 1e-4; 
simulation_parameters.output_all_maxima = false; 
simulation_parameters.compute_std_errors = false;
simulation_parameters.boot_samples = 0;
simulation_parameters.lambda_fct    = lambda;
simulation_parameters.lambda_stderr = lambda;

significant_spurts_data = [];

ncase = size(subject_ids,1);  % number of curves (cases) in the raw data
for id=1:ncase
    % compute spurts data
    [spurt_data graph_handle] = find_spurts(simulation_parameters, id);
    
    % append to existing data
    significant_spurts_data = [significant_spurts_data spurt_data];

    % output plot for individual data
    group = simulation_parameters.group_ids(id);  % retrieving group for this case
    fileName = [resultsDir, 'case_', num2str(simulation_parameters.subject_ids(id)), '_' ...
        char(simulation_parameters.group_labels(group)), '.png'];
    print('-f', '-dpng',  fileName);
    close
end

% output spurts data to file as text
data_file_name = [resultsDir, 'AMD_spurts.txt'];
output_significant_data(data_file_name, significant_spurts_data, ...
    simulation_parameters.compute_std_errors);

% plot group data
h = plot_spurts(significant_spurts_data, simulation_parameters, SORT_CASE_ID);
if (h)
    print('-f', '-dpng',  [resultsDir,'spurts.png']);
    close
end
