function [ encodedVect ] = encode( buffer, letterCount, lowTarget, highTarget, debug )
    % Description: Converts a vector of characters to a vector of binary values
    % using local (one-of-N) coding.
    %
    % Inputs:
    % 1. buffer: a vector of characters (ASCII-coded)
    % 2. letterCount: number of accentuated letters to be encoded
    %    (for english, use 26 letters)
    % 3. lowTarget: value indicating absence of letter (ex. 0)
    % 4. highTarget: value indicating presence of letter (ex. 1)
    % 5. debug: boolean value. If true, uses verbose mode
    %
    % Outputs:
    % 1. encodedVect: a vector of local-encoded values 
    % e.g., letter A = 1 0 0 0 ...; B = 0 1 0 0 ...
    %
    %
    % (c) 2011 Frdric Dandurand
    %

    slots = size(buffer,2);  % number of slot positions in the buffer
    % set all values to lowTarget by default (override will occur where
    % necessary)
    encodedVect = lowTarget * ones(letterCount, slots);
    for slot=1:slots
        if (buffer(slot) == 0)
            % When code nothing for spaces (vector of zeros)
            % which means ignore blanks
            encodedVect(:,slot) = zeros(letterCount,1);
        elseif (buffer(slot) >= 'a' && buffer(slot) <= 'z')
            % standard ascii character between a and z
            % position in slot corresponds to a=1,...z=26
            letterPos = 1 + buffer(slot)-'a';
            % use the highTarget value to code letter
            encodedVect(letterPos, slot) = highTarget;
        else
            % special characters, coded above position 26 in vector
            switch (buffer(slot))
                case ''
                    letterPos = 27;
                case ''
                    letterPos = 28;
                case ''
                    letterPos = 29;
                case ''
                    letterPos = 30;
                case ''
                    letterPos = 31;
                case ''
                    letterPos = 32;
                case ''
                    letterPos = 33;
                case ''
                    letterPos = 34;
                case ''
                    letterPos = 35;
                case ''
                    letterPos = 36;
                case ''
                    letterPos = 37;
                otherwise
                    error('unknown char')
            end
            % use the highTarget value to code letter
            encodedVect(letterPos, slot) = highTarget;
        end
    end
    
    if (debug)
        % output content of encodedVect plus 3 heading zeros (for spaces)
        tmp = [zeros(letterCount,3) num2str(encodedVect)];
        
        % replace first column with the letter value (only works for a to z)
        for let=1:letterCount
            tmp(let,1) = 'a' + let - 1;
        end
        disp(tmp)
    end
end

