function [ formattedStr ] = dispStats(trainData, iter)
    % Description: generates a formatted string of the training progress
    % for a given iteration. Information displayed is the SOM map with the
    % best matching unit for each pattern, the number of distinct best
    % matching units, the total distance and activation of best matching
    % units
    %
    % Inputs:
    % 1. trainData : a structure that contains training data information
    % required fields:
    %     1.1. inputMap : indicates which map was used as input for SOM
    %     training
    %     1.2. mapSize: two-element vector of input map size
    %     1.3. useDistance: boolean flag indicating method for selecting 
    %          best matching units (distance = 1, activation = 0)
    %     1.4. trainingResults: array (one element per iteration)
    %        1.4.1. winnerVal: total SOM distance for all letters 
    %        1.4.2. worstLetter: label of the worst letter (smallest
    %        activation or largest distance, depending on useDistance flag)
    %        1.4.3. worstValue: distance or activation of the worst letter
    %        1.4.4. winnerLocation: structure of one element per letter
    %        with information about the bst matching unit (BMU)
    %          1.4.4.1. id: letter label
    %          1.4.4.2. row: row position of the BMU
    %          1.4.4.3. col: column position of the BMU
    %        1.4.5. bestMatchingUnitsCount: number of distinct best
    %        matching units on the SOM map
    % 2. iter : current iteration for which information should be outputted
    %
    % Outputs:
    % 1. formattedStr: string to be displayed or saved in a text file
    %
    %
    % (c) 2011 Frdric Dandurand
    %
    dim3=trainData.mapSize(1);
    dim4=trainData.mapSize(2);
    
    formattedStr = ['Iteration : ', num2str(iter), sprintf('\n')];
    
    for i=1:dim3
        for j=1:dim4
            % For each unit on the SOM map
            count = 0;  % keeping track of how many times this unit was a BMU
            foundLabel = '';  % letter label for this BMU
            for patId=1:length(trainData.trainingResults(iter).winnerLocation)
                % for each pattern, seek the ones that correspond to the
                % current unit
                if((trainData.trainingResults(iter).winnerLocation(patId).row == i) && ...
                   (trainData.trainingResults(iter).winnerLocation(patId).col == j))
                    % BMU for this pattern corresponds to the current unit
                    count = count + 1;  % increase count of BMU for this unit
                    foundLabel = trainData.trainingResults(iter).winnerLocation(patId).id; % store last letter label
                end
            end
            if (count == 0)
                formattedStr = [formattedStr, '...'];   % unit is not a BMU
            elseif (count == 1)
                % unit is a BMU for a single letter - show which letter on
                % the display
                formattedStr = [formattedStr, sprintf('%s%i', foundLabel), '..'];
            else
                % unit is a BMU for mutiple letters - show one letter and a
                % + sign to indicate there are more letters whose BMU is
                % the current unit
                formattedStr = [formattedStr, sprintf('%s%i', foundLabel), '+', num2str(count)];
            end
            formattedStr = [formattedStr, ' '];
        end
        formattedStr = [formattedStr, sprintf('\n')];
    end
    
    [x y bmuVect] = find([trainData.trainingResults(1:iter).bestMatchingUnitsCount]);  % bmuVect is the vector of BMU count for all iterations
    [bmuVal bmuPos] = max(bmuVect);  % returns the value and the position of the maximum (first time encountered in the vector)
    bmuAgo = iter - bmuPos;  % how many iterations ago was the maximum reached
    formattedStr = [formattedStr, 'Number of distinct best matching units = ', num2str(trainData.trainingResults(iter).bestMatchingUnitsCount, '%i'), ' (Max ', ...
            num2str(bmuVal, '%i'), ' found ', int2str(bmuAgo), ' iteration(s) ago)', sprintf('\n')];

    if (trainData.useDistance)
        [x y distVect] = find([trainData.trainingResults(1:iter).winnerVal]);  % distVect is the vector of distances for all iterations
        [distVal distPos] = min(distVect); % returns the value and the position of the minimum (first time encountered in the vector)
        distAgo = iter - distPos;
        formattedStr = [formattedStr, 'Total distance of winners = ', num2str(trainData.trainingResults(iter).winnerVal, '%5.5f'), ' (Min ', ...
            num2str(distVal, '%5.5f'), ' found ', int2str(distAgo), ' iteration(s) ago)', sprintf('\n')];
        [x y worstVect] = find([trainData.trainingResults(1:iter).worstValue]);
        [worstVal worstPos] = min(worstVect);
    else
        [x y actVect] = find([trainData.trainingResults(1:iter).winnerVal]);
        [actVal actPos] = max(actVect);
        actAgo = iter - actPos;
        formattedStr = [formattedStr, 'Total activation of winners = ', num2str(trainData.trainingResults(iter).winnerVal, '%5.5f'), ' (Max ', ...
            num2str(actVal, '%5.5f'), ' found ', int2str(actAgo), ' iteration(s) ago)', sprintf('\n')];
        [x y worstVect] = find([trainData.trainingResults(1:iter).worstValue]);
        [worstVal worstPos] = max(worstVect);
    end

    worstAgo = iter - worstPos;
    formattedStr = [formattedStr, 'Worst letter = ', num2str(trainData.trainingResults(iter).worstValue, '%5.5f'),  ...
        ' (Best ', num2str(worstVal, '%5.5f'), ' found ', int2str(worstAgo), ' iteration(s) ago: ', trainData.trainingResults(iter).worstLetter, ')', sprintf('\n\n')];
end
