function [ angles ] = extractAngles (map)
    % Description: computes angles on the map made by line segments of 3
    % pixels in each of 28 possible cases:
    % Case no 1:
    %  **X*X
    %  **XX*
    %  **X**
    %  *****
    %  *****
    % Vertical at x, y-1 and forward slash diagonal (fd) at x+1, y-1

    % Case no 2:
    %  **X**
    %  **X**
    %  **XXX
    %  *****
    %  *****
    % Vertical at x, y-1 and horizontal x+1, y

    % Case no 3:
    %  **X**
    %  **X**
    %  **X**
    %  ***X*
    %  ****X
    % Vertical at x, y-1 and backslash diagonal (bd) x+1, y+1

    % Case no 4:
    %  **X**
    %  **X**
    %  **X**
    %  **X**
    %  **X**
    % Vertical at x, y-1 and vertical x, y+1

    % Case no 5:
    %  **X**
    %  **X**
    %  **X**
    %  *X***
    %  X****
    % Vertical at x, y-1 and fd x-1, y+1

    % Case no 6:
    %  **X**
    %  **X**
    %  XXX**
    %  *****
    %  *****
    % Vertical at x, y-1 and horiz x-1, y

    % Case no 7:
    %  X*X**
    %  *XX**
    %  **X**
    %  *****
    %  *****
    % Vertical at x, y-1 and bd x-1, y-1


    % Case no 8:
    %  ****X
    %  ***X*
    %  **XXX
    %  *****
    %  *****

    % Case no 9:
    %  ****X
    %  ***X*
    %  **X**
    %  ***X*
    %  ****X

    % Case no 10:
    %  ****X
    %  ***X*
    %  **X**
    %  **X**
    %  **X**

    % Case no 11:
    %  ****X
    %  ***X*
    %  **X**
    %  *X***
    %  X****

    % Case no 12:
    %  ****X
    %  ***X*
    %  XXX**
    %  *****
    %  *****

    % Case no 13:
    %  X***X
    %  *X*X*
    %  **X**
    %  *****
    %  *****

    % Case no 14:
    %  *****
    %  *****
    %  **XXX
    %  ***X*
    %  ****X

    % Case no 15:
    %  *****
    %  *****
    %  **XXX
    %  **X**
    %  **X**

    % Case no 16:
    %  *****
    %  *****
    %  **XXX
    %  *X***
    %  X****

    % Case no 17:
    %  *****
    %  *****
    %  XXXXX
    %  *****
    %  *****


    % Case no 18:
    %  X****
    %  *X***
    %  **XXX
    %  *****
    %  *****


    % Case no 19:
    %  *****
    %  *****
    %  **X**
    %  **XX*
    %  **X*X

    % Case no 20:
    %  *****
    %  *****
    %  **X**
    %  *X*X*
    %  X***X

    % Case no 21:
    %  *****
    %  *****
    %  XXX**
    %  ***X*
    %  ****X

    % Case no 22:
    %  X****
    %  *X***
    %  **X**
    %  ***X*
    %  ****X

    % Case no 23:
    %  *****
    %  *****
    %  **X**
    %  *XX**
    %  X*X**

    % Case no 24:
    %  *****
    %  *****
    %  XXX**
    %  **X**
    %  **X**

    % Case no 25:
    %  X****
    %  *X***
    %  **X**
    %  **X**
    %  **X**


    % Case no 26:
    %  *****
    %  *****
    %  XXX**
    %  *X***
    %  X****


    % Case no 27:
    %  X****
    %  *X***
    %  **X**
    %  *X***
    %  X****


    % Case no 28:
    %  X****
    %  *X***
    %  XXX**
    %  *****
    %  *****

    % table for how to distinguish cases
    % 12:00 --> (x,   y-1)
    %  1:30 --> (x+1, y-1)
    %  3:00 --> (x+1, y)
    %  4:30 --> (x+1, y+1)
    %  6:00 --> (x,   y+1)
    %  7:30 --> (x-1, y+1)
    %  9:00 --> (x-1, y)
    % 10:30 --> (x-1, y-1)

    %  Traits: 
    % Horizontal: -
    % Vertical: |
    % Forward slash: /
    % Back slash: \

    % CaseId    What appears at coordinate
    %           12:00 1:30 3:00 4:30 6:00 7:30 9:00 10:30
    %  1          |     /
    %  2          |          -
    %  3          |               \
    %  4          |                   |
    %  5          |                         /
    %  6          |                             -
    %  7          |                                   \
    %  8                /   -
    %  9                /        \
    %  10               /             |
    %  11               /                   /
    %  12               /                       -
    %  13               /                             \
    %  14                   -    \
    %  15                   -         |
    %  16                   -               /
    %  17                   -                   -
    %  18                   -                         \
    %  19                        \    |
    %  20                        \          /
    %  21                        \              -
    %  22                        \                    \
    %  23                             |     /
    %  24                             |         -
    %  25                             |               \
    %  26                                   /   -
    %  27                                   /         \
    %  28                                       -     \
    
    % outputs:
    % 1. angles: a 3D matrix of the same size as map for the first 2
    % dimensions (topological) and with a third dimension, a cortical
    % column, that codes for the type of angle detected at the point on the
    % map.
    %
    % (c) 2011 Frdric Dandurand
    %

    lineConstants;
    
    % bigMap is used to avoid edge problems. 
    %
    bigMap = zeros(size(map,1)+2, size(map,2)+2,size(map,3));
    bigMap(2:1+size(map,1), 2:1+size(map,2), :) = map;

    angles = zeros(size(map,1), size(map,2), 28);
    
        for row=2:size(bigMap,1)-1
            for col=2:size(bigMap,2)-1
                % computing line segments
                
                % Naming convention is as follows: LHour, e.g., L12 means a
                % line at 12:00 (which has to be vertical

                % L12  = vert at  (x,   y-1)
                % L1.5 = fs at    (x+1, y-1)
                % L3   = horiz at (x+1, y)
                % L4.5 = bs at    (x+1, y+1)
                % L6   = vert at  (x,   y+1)
                % L7.5 = fs at    (x-1, y+1)
                % L9   = horiz at (x-1, y)
                % L10.5= bs at    (x-1, y-1)
                L12   = bigMap(row,   col-1, VERT_INDEX);
                L1    = bigMap(row+1, col-1, FS_INDEX);
                L3    = bigMap(row+1, col,   HORIZ_INDEX);
                L4    = bigMap(row+1, col+1, BS_INDEX);
                L6    = bigMap(row,   col+1, VERT_INDEX);
                L7    = bigMap(row-1, col+1, FS_INDEX);
                L9    = bigMap(row-1, col,   HORIZ_INDEX);
                L10   = bigMap(row-1, col-1, BS_INDEX);
                
                c= zeros(28,1);

                % computing angles as the product (conjunction) of two
                % line segments appropriately oriented
                %           12:00 1:30 3:00 4:30 6:00 7:30 9:00 10:30
                %  1          |     /
                c(1) = L12 * L1;
                %  2          |          -
                c(2) = L12 * L3;
                %  3          |               \
                c(3) = L12 * L4;
                %  4          |                   |
                c(4) = L12 * L6;
                %  5          |                         /
                c(5) = L12 * L7;
                %  6          |                             -
                c(6) = L12 * L9;
                %  7          |                                   \
                c(7) = L12 * L10;
                %  8                /   -
                c(8) = L1 * L3;
                %  9                /        \
                c(9) = L1 * L4;
                %  10               /             |
                c(10) = L1 * L6; 
                %  11               /                   /
                c(11) = L1 * L7;
                %  12               /                       -
                c(12) = L1 * L9;
                %  13               /                             \
                c(13) = L1 * L10;
                %  14                   -    \
                c(14) = L3 * L4;
                %  15                   -         |
                c(15) = L3 * L6;
                %  16                   -               /
                c(16) = L3 * L7;
                %  17                   -                   -
                c(17) = L3 * L9;
                %  18                   -                         \
                c(18) = L3 * L10;
                %  19                        \    |
                c(19) = L4 * L6;
                %  20                        \          /
                c(20) = L4 * L7;
                %  21                        \              -
                c(21) = L4 * L9;
                %  22                        \                    \
                c(22) = L4 * L10;
                %  23                             |     /
                c(23) = L6 * L7;
                %  24                             |         -
                c(24) = L6 * L9;
                %  25                             |               \
                c(25) = L6 * L10;
                %  26                                   /   -
                c(26) = L7 * L9;
                %  27                                   /         \
                c(27) = L7 * L10;
                %  28                                       -     \
                c(28) = L9 * L10;
                
                % storing cortical column of angles at row, col
                angles(row-1,col-1,:) = c;
            end
        end
    end
