function [activations, maxActVal, maxActRow, maxActCol] = computeSOMActivations( pat, weights, weightsMask, debug )
    % Description: computes activations of the SOM units given inputs 
    % (product of input values and connection weights). 
    % Outputs the unit with maximal activation (best matching unit). 
    %
    % Inputs:
    % 1. pat : input pattern
    % 2. weights : matrix containing connection weight values
    % (size = dim1 x dim2 x CORTICAL_COLS x dim3 x dim4)
    % 3. weightsMask : matrix containing receptive field connectivity
    % (size = dim1 x dim2 x CORTICAL_COLS x dim3 x dim4)
    % 4. debug : flag for debug mode
    %
    % Outputs:
    % 1. activations : matrix of activations for each of the SOM units 
    % (size = dim3 x dim4) summed accross all inputs rows, cols and
    % cortical columns.
    % 2. maxActVal : activation of the best matching unit
    % 3. maxActRow : row position on the SOM map of best matching unit
    % 4. maxActCol : col position on the SOM map of best matching unit
    %
    %
    % (c) 2011 Frdric Dandurand
    %

    [r1 c1 a r2 c2] = size(weights);
        
    % repeating inputs for each SOM unit
    patMat = repmat(pat, [1 1 1 r2 c2]);
    act = patMat .* weights .* weightsMask;
    % total activation is the sum across all inputs of the 
    % input value (patMat) and connection weight, under the constraint
    % of receptive fields (weightsMask).
    activations = squeeze(sum(sum(sum(act, 1), 2), 3));
    
    % find unit that is maximally activated
    maxActVal = max(max(activations));
    [maxActRow, maxActCol] = find(activations == maxActVal);

    if (debug)
        % in debug mode: verify that inputs are compatible in size with the
        % connection weights
        if (r1 ~= size(pat,1) || c1 ~= size(pat,2) || a ~= size(pat,3))
            error('computeSOMActivations error: pat and weights have incompatible sizes');
        end

        activations2 = zeros(r2, c2);

        % computing activations using a for loop method
        for toRow=1:r2
            for toCol=1:c2
                for fromRow = 1:r1
                    for fromCol = 1:c1
                        for cortical_col = 1:a
                            % verify if point is in the receptive field
                            if (weightsMask(fromRow, fromCol, cortical_col, toRow, toCol) == 1) 
                                activations2(toRow, toCol) = activations2(toRow, toCol) + ...
                                    pat(fromRow, fromCol, cortical_col) * weights(fromRow, fromCol, cortical_col, toRow, toCol);
                            end
                        end
                    end
                end
            end
        end
        
        % make sure methods are consistent (differences are within a small
        % tolerance)
        testDiff(activations, activations2)
    end
end
