function [ distances, minDistVal, minDistRow, minDistCol ] = computeSOMDistances( pat, weights, weightsMask, debug )
    % Description: computes distances between input pattern of weight
    % vectors connected to each SOM unit. Outputs the unit with minimal
    % distance (best matching unit). 
    %
    % Inputs:
    % 1. pat : input pattern
    % 2. weights : matrix containing connection weight values
    % (size = dim1 x dim2 x CORTICAL_COLS x dim3 x dim4)
    % 3. weightsMask : matrix containing receptive field connectivity
    % (size = dim1 x dim2 x CORTICAL_COLS x dim3 x dim4)
    % 4. debug : flag for debug mode
    %
    % Outputs:
    % 1. distances : matrix of distances for each of the SOM units 
    % (size = dim3 x dim4) summed accross all inputs rows, cols and
    % cortical columns.
    % 2. minDistVal : distance of the best matching unit from inputs
    % 3. minDistRow : row position on the SOM map of best matching unit
    % 4. minDistCol : col position on the SOM map of best matching unit
    %
    %
    % (c) 2011 Frdric Dandurand
    %

    [r1 c1 a r2 c2] = size(weights);

    % repeating inputs for each SOM unit
    patMat = repmat(pat, [1 1 1 r2 c2]);
    
    % computing distances between inputs and connection weights
    % applying the constraint of receptive fields
    dist = weightsMask .* (patMat - weights);
    dist = power(dist, 2);
    % total distance is the sum of squared distances
    distances = squeeze(sum(sum(sum(dist, 1), 2), 3));
    
    % find unit with shortest distance
    minDistVal = min(min(distances));
    [minDistRow, minDistCol] = find(distances == minDistVal);
    
    if (debug)
        % in debug mode: verify that inputs are compatible in size with the
        % connection weights
        if (r1 ~= size(pat,1) || c1 ~= size(pat,2) || a ~= size(pat,3))
            error('computeSOMDistances error: pat and weights have incompatible sizes');
        end

        distances2 = zeros(r2, c2);

        % computing distances with for loops
        for toRow=1:r2
            for toCol=1:c2
                for fromRow = 1:r1
                    for fromCol = 1:c1
                        for cortical_col = 1:a
                            % verify if point is in the receptive field
                            if (weightsMask(fromRow, fromCol,  cortical_col, toRow, toCol) == 1)
                                % updating distance for current SOM rows
                                % and cols with the square of difference
                                % between input pattern and connection
                                % weights
                                distances2(toRow, toCol) = distances2(toRow, toCol) + ...
                                    power(pat(fromRow, fromCol, cortical_col) - weights(fromRow, fromCol, cortical_col, toRow, toCol), 2);
                            end
                        end
                    end
                end
            end
        end
        
        % make sure methods are consistent (differences are within a small
        % tolerance)
        testDiff(distances, distances2);

    end
end
