function [ updatedWeights ] = computeUpdatedWeights(weights,weightsMask,pat,...
    dim1,dim2,CORTICAL_COLS,dim3,dim4,...
    winnerRow,winnerCol,iterId,learningRate,temperatureFactor,factorVector,debug)
    %
    % Description: computes updated SOM weights
    % 
    % Inputs:
    % 1. weights : current SOM weights (size = dim1 x dim2 x CORTICAL_COLS
    % x dim3 x dim 4)
    % 2. weightsMask : weight mask of the receptive field
    % 3. pat : input pattern
    % 4. dim1 : number of rows of the input layer
    % 5. dim2 : number of columns of the input layer
    % 6. CORTICAL_COLS : number of cortical columns in the input layer
    % 7. dim3 : number of rows of the SOM layer
    % 8. dim4 : number of columns of the input layer
    % 9. winnerRow : row position of the SOM winner unit
    % 10. winnerCol : column position of the SOM winner unit
    % 11. iterId : learning iteration (relevant only with non-zero
    % temperature)
    % 12. learningRate : SOM learning rate 
    % 13. temperatureFactor: exponential decay of factors over iterations
    % (makes learning rates decrease over time)
    % 14. factorVector : vector of 4 values giving the contributions at the
    % target, and at city block distances (Minkovski) of 1, 2 and 3 from
    % the target
    % 15. debug : flag for debug mode
    %
    % Outputs:
    % 1. updatedWeights : updated SOM weights (size = dim1 x dim2 x CORTICAL_COLS
    % x dim3 x dim 4)
    %
    %
    % (c) 2011 Frdric Dandurand
    %
    
    % compute the matrix that indicates by what factor each unit of the SOM 
    % should be updated. The value is maximal at the winning unit, and
    % the neighbours are also allowed to learn according to factorVector
    learningFactor(1,1,1, :, :) = distFactorMat(winnerRow,winnerCol,dim3,dim4,iterId,learningRate,temperatureFactor,factorVector);

    % replicating the matrices (inputs and weights to update) over every input and cortical column
    learningFactorMat = repmat(learningFactor, [dim1 dim2 CORTICAL_COLS 1 1]);
    patMat = repmat(pat, [1 1 1 dim3 dim4]);

    % update weights with the difference between current weights and
    % inputs (updatedWeights - patMat) scaled by the learning matrix
    % (learningFactorMat) and only for relevant items in weightsMask
    updatedWeights = weights - weightsMask .* learningFactorMat .* (weights - patMat);
    
    if (debug)
        % in debug mode, compare updatedWeights computed with the fast
        % method and updatedWeights2, computed here, using a clearer but
        % slower method involving for loops
        updatedWeights2 = weights;

        % decompose the update by the 4 dimensions (input rows, input cols,
        % SOM rows, SOM cols) and the cortical columns
        for toRow=1:dim3
            for toCol=1:dim4
                learningFactor = distFactor(toRow,toCol,winnerRow,winnerCol,iterId,learningRate,temperatureFactor,factorVector);
                if (learningFactor > 0)
                    % proceed further only when the learning factor is
                    % non-zero (that is, for SOM units that need to be
                    % trained)
                    for fromRow=1:dim1
                        for fromCol=1:dim2
                            for cortCol=1:CORTICAL_COLS
                                % is this connection within the receptive field?
                                if (weightsMask(fromRow,fromCol,cortCol,toRow,toCol) == 1)
                                    % yes, so update weights with the difference between current weights and
                                    % inputs (updatedWeights2 - pat) scaled by the learning matrix
                                    % (learningFactor)
                                    updatedWeights2(fromRow,fromCol,cortCol,toRow,toCol) = updatedWeights2(fromRow,fromCol,cortCol,toRow,toCol) - ...
                                        learningFactor * (updatedWeights2(fromRow,fromCol,cortCol,toRow,toCol) - pat(fromRow,fromCol,cortCol)); 
                                end
                            end
                        end
                    end
                end
            end
        end

        % verify that the methods yields the same result (within tolerance)
        testDiff(updatedWeights, updatedWeights2);
    end
end

