function [ factor ] = distFactor( row1, col1, row2, col2, iter, learningRate, temperatureFactor, factorVector )
    % Description: distFactor computes a multiplicative factor that is function of the
    % distance between two points (row1, col1) and (row2, col2).
    %
    % Distances used are city block distances (Minkovski):
    % e.g., 
    % 5 4 3 4 5 6
    % 4 3 2 3 4 5
    % 3 2 1 2 3 4
    % 2 1 0 1 2 3
    % 3 2 1 2 3 4
    % 4 3 2 3 4 5
    % 5 4 3 4 5 6
    %
    % Inputs:
    % 1. row1 : row position of the first point
    % 2. col1 : column position of the first point
    % 3. row2 : row position of the second point
    % 4. col2 : column position of the second point
    % 5. iter : learning iteration (relevant only with non-zero
    % temperature)
    % 6. learningRate : SOM learning rate 
    % 7. temperatureFactor : exponential decay of factors over iterations
    % (makes learning rates decrease over time)
    % 8. factorVector : vector of 4 values giving the contributions at the
    % target, and at city block distances (Minkovski) of 1, 2 and 3 from
    % the target
    % 
    % Outputs:
    % 1. factor : a real value giving the multiplicative factor SOM learning 
    %
    %
    % (c) 2011 Frdric Dandurand
    %

    % calculating the city block distance (Minkovski) between the two
    % points
    dist = abs(row1-row2) + abs(col1-col2);
   
    if (dist > 3)
        % distances farther than 3 are outside the neighbourhood
        factor = 0;
    else
        % distances within neighbourhood, use correct factor and scale by
        % learning rate
        factor = factorVector(dist+1)*learningRate;
    end
        
    % exponential attenuation by the temperature factor (no attention if
    % factor = 0)
    attenuation = exp(-iter*temperatureFactor);
    factor = factor * attenuation;
end
