function [ factorMat ] = distFactorMat( targetRow, targetCol, totalRow, totalCol, ...
    iter, learningRate, temperatureFactor, factorVector )
    % Description: computes a matrix of multiplicative factors for 
    % neighbourhood of target point (targetRow, targetCol)
    %
    % Distances used are city block distances (Minkovski):
    % e.g., 
    % 5 4 3 4 5 6
    % 4 3 2 3 4 5
    % 3 2 1 2 3 4
    % 2 1 0 1 2 3
    % 3 2 1 2 3 4
    % 4 3 2 3 4 5
    % 5 4 3 4 5 6
    %
    % Inputs:
    % 1. targetRow : row position of the target 
    % 2. targetCol : column position of the target
    % 3. totalRow : number of rows
    % 4. totalCol : number of columns
    % 5. iter : learning iteration (relevant only with non-zero
    % temperature)
    % 6. learningRate : SOM learning rate 
    % 7. temperatureFactor : exponential decay of factors over iterations
    % (makes learning rates decrease over time)
    % 8. factorVector : vector of 4 values giving the contributions at the
    % target, and at city block distances (Minkovski) of 1, 2 and 3 from
    % the target
    % 
    % Outputs:
    % 1. factorMat : a matrix of size totalRow x totalCol of neighbourhood 
    % multiplicative factors for SOM learning at point (targetRow,
    % targetCol)
    %
    %
    % (c) 2011 Frdric Dandurand
    %

    factorMat = zeros(totalRow, totalCol);
    
    % Distance 3:
    dist = 3;
    factorMat(max(1,targetRow-dist):min(totalRow, targetRow+dist), targetCol) = factorVector(dist+1);
    factorMat(targetRow, max(1,targetCol-dist):min(totalCol, targetCol+dist)) = factorVector(dist+1);
    factorMat(max(1,targetRow-1):min(totalRow, targetRow+1), max(1,targetCol-2):min(totalCol, targetCol+2)) = factorVector(dist+1);
    factorMat(max(1,targetRow-2):min(totalRow, targetRow+2), max(1,targetCol-1):min(totalCol, targetCol+1)) = factorVector(dist+1);

    % Distance 2:
    dist = 2;
    factorMat(max(1,targetRow-dist):min(totalRow, targetRow+dist), targetCol) = factorVector(dist+1);
    factorMat(targetRow, max(1,targetCol-dist):min(totalCol, targetCol+dist)) = factorVector(dist+1);
    factorMat(max(1,targetRow-1):min(totalRow, targetRow+1), max(1,targetCol-1):min(totalCol, targetCol+1)) = factorVector(dist+1);

    % Distance 1:
    dist = 1;
    factorMat(max(1,targetRow-dist):min(totalRow, targetRow+dist), targetCol) = factorVector(dist+1);
    factorMat(targetRow, max(1,targetCol-dist):min(totalCol, targetCol+dist)) = factorVector(dist+1);

    % target:
    dist = 0;
    factorMat(targetRow, targetCol) = factorVector(dist+1);

    % multiplying by the learning rate
    factorMat = factorMat * learningRate;
    
    % exponential attenuation by the temperature factor (no attention if
    % factor = 0)
    attenuation = exp(-iter*temperatureFactor);
    factorMat = factorMat * attenuation;
end
