function [ weightsMask ] = getWeightsMask( dim1, dim2, CORTICAL_COLS, dim3, dim4, receptiveFieldSize )
    % Description: generates the mask for SOM connection weights
    % The input layer has dim1 rows, dim2 columns of CORTICAL_COLS values
    % The SOM layer has dim3 rows and dim4 columns
    % receptiveFieldSize is a vector of two values indicating how many
    % input rows and columns are visible from the SOM.
    % 
    % Inputs: 
    % 1. dim1 : number of rows of the input layer
    % 2. dim2 : number of columns of the input layer
    % 3. CORTICAL_COLS : number of cortical columns in the input layer
    % 4. dim3 : number of rows of the SOM layer
    % 5. dim4 : number of columns of the input layer
    % 6. receptiveFieldSize : size of the receptive field of inputs visible from the SOM layer
    %
    % Outputs:
    % 1. weightsMask: a matrix of size = dim1 x dim2 x CORTICAL_COLS x dim3 x dim4
    % that contains values: 
    % 0 = unused connection weights (not in receptive fields)
    % 1 = connection weights within receptive field
    %
    %
    % (c) 2011 Frdric Dandurand
    %

    weightsMask = zeros(dim1, dim2, CORTICAL_COLS, dim3, dim4);  % location of active connections are indicated by values of one (1), and units not connected = 0

    % now, initialize connections of elements within receptive field
    % for each row and column of the SOM map
    for toRow = 1:dim3
        for toCol = 1:dim4
            % Find, on the input map, which unit is aligned with the SOM map unit
            % assuming that the two maps are stretched to cover the same area
            fromRow = round(toRow * dim1 / dim3);
            fromCol = round(toCol * dim2 / dim4);

            % Determine the coordinates of the visible square on the input map, 
            % centered around (fromRow,fromCol).
            row2min = max(fromRow-(receptiveFieldSize(1)-1)/2,1);
            row2max = min(fromRow+(receptiveFieldSize(1)-1)/2, dim1);
            col2min = max(fromCol-(receptiveFieldSize(2)-1)/2, 1);
            col2max = min(fromCol+(receptiveFieldSize(2)-1)/2, dim2);

            % those connections are within the receptive field
            weightsMask(row2min:row2max, col2min:col2max, :, toRow, toCol) = 1;
        end
    end
end

