function testDiff( matrix1, matrix2 )
    % Description: tests that all values in two matrices are within some small
    % tolerance value. Stops the program with an error if a large
    % difference is found. Designed for debugging purposes, especially when
    % two computation methods are used, and we want to make sure they yield
    % consistent results within some tolerance (e.g., numerical errors).
    %
    % Inputs:
    % 1. matrix1 : first matrix to compare
    % 2. matrix2 : second matrix to compare
    %
    %
    % (c) 2011 Frdric Dandurand
    %

    % first, make sure the matrices have the same size
    if (size(matrix1) ~= size(matrix2))
      disp(['Matrix 1 = ', num2str(size(matrix1)), '  Matrix 2 = ', num2str(size(matrix2))])
      error('sizes different')
    end

    tolerance = 1e-5;  % numerical difference tolerated
    % this tolerance is applied to the normalized difference between
    % matrices. Normalization factor is the largest value in both the
    % matrices

    % seeking the maximal value in the first matrix
    m1=matrix1;
    for dim=1:size(size(matrix1),2)
      % across all dimensions, look for maximum
      m1 = squeeze(max(m1));
    end
    % seeking the maximal value in the second matrix
    m2=matrix2;
    for dim=1:size(size(matrix2),2)
      % across all dimensions, look for maximum
      m2 = squeeze(max(m2));
    end
    % take the maximum of the two matrices as normalization factor
    m = max(m1,m2);
    
    % compute normalized differences
    diff = abs(matrix1-matrix2)/m;
    
    % extract the maximal normalized difference
    maxDiff = diff;
    for dim=1:size(size(diff),2)
      % across all dimensions, look for maximum
      maxDiff = squeeze(max(maxDiff));
    end

    % exit in error if the difference is larger than allowed tolerance
    if (maxDiff > tolerance)
    disp(['Max value = ', num2str(maxDiff)])
    error('testDiff failed')
    end
end
